var util = require('util'),
    stream = require('stream');

var RECTYPE = {
    DATA: 0x00,             // <256 bytes (typically 16 or 32)
    EOF: 0x01,              // 0 bytes
    SEGMENT_ADDR: 0x02,     // 2 bytes (<< 4 + address, we don't use)
    SEGMENT_START: 0x03,    // 4 bytes (CS:IP, we don't use)
    LINEAR_ADDR: 0x04,      // 2 bytes (<< 16 + address)
    LINEAR_START: 0x05      // 4 bytes (EIP, we don't use)
};

var RANGES = {
    misc: {start:0x00000000, end:0x00005c00, wdsz:2},               // NONPUBLIC
    flash: {start:0x00006000, end:0x0002A800, wdsz:2},
    config: {start:0x0002A800, end:0x0002AC00, wdsz:2},             // NONPUBLIC
    misc2: {start:0x0002AC00, end:0x00080000, wdsz:2},              // NONPUBLIC
    eeprom: {start:0x00100000, end:0x001F0000, wdsz:1},
    initVec: {start:0xFFFFFF00, end:0xFFFFFF10, wdsz:1},            // NONPUBLIC (aligned to page, but only 16 bytes needed)
    // NONPUBLIC NOTE: we use this alias in hex_reader just for a bit of obfuscation
    checksum: {start:0xFFFFFF00, end:0xFFFFFF10, wdsz:1},
    _factory: {start:0x00005c00, end:0x00006000, wdsz:2},           // NONPUBLIC (this one is given special behaviour on MCU)
};

// HACK: expose for structsInfo documentation helper                // NONPUBLIC
exports._structs = {RANGES:RANGES};                                 // NONPUBLIC

function _addrInRange(addr, name) {
    var range = RANGES[name],
        rAddr = addr / range.wdsz;      // match stupid Microchip programmer
    return (range && rAddr >= range.start && rAddr < range.end) ? range : null;
}

function _rangeForAddr(addr) {
    var range = null;
    Object.keys(RANGES).forEach(function (name) {
        if (!range) range = _addrInRange(addr, name);
    }, this);
    return range;
}

function _chiptuneSum(a,b) { return (a+b) & 0xFF; }     // 8-bit addition
function _hex(n,ff) { return (n+ff+1).toString(16).slice(1); }
function _unhex(s) { return parseInt(s,16); }

function checksum(vals, data) {
    var chk1 = vals.reduce(_chiptuneSum, 0),
        chk2 = Array.prototype.reduce.call(data, _chiptuneSum, 0);
    return _chiptuneSum(0x100, -_chiptuneSum(chk1,chk2));
}


function HexReader(opts) {
    opts || (opts = {});
    stream.Transform.call(this, {objectMode:true});
    this._line = '';
    this._addr = 0;      // active linear address prefix
    this._nxtA = -1;     // next (expected) address, if not match emit new start address
    this._wdsz = opts.bytesPerAddr || 1;    // NOTE: Microchip's bootloader always uses 1-byte addressing *in HEX*
}
util.inherits(HexReader, stream.Transform);

HexReader.prototype._processLine = function (str) {
    var line = readLine(str);
    switch (line.rectype) {
        case RECTYPE.DATA:
            var startAddress = this._addr + line.offset;
            if (startAddress !== this._nxtA) this.push({startAddress:startAddress});
            this.push(line.data);
            this._nxtA = startAddress + line.data.length * this._wdsz;
            break;
        case RECTYPE.SEGMENT_ADDR:
            this._addr = line.data.readUInt16BE(0) << 4;
            break;
        case RECTYPE.LINEAR_ADDR:
            this._addr = line.data.readUInt16BE(0) * Math.pow(2, 16);            // (avoid signed shift operator)
            break;
        case RECTYPE.EOF:
            this.push(null);
            break;
    }
};

HexReader.prototype._processLines = function (lines, cb) {
    try {
        lines.forEach(HexReader.prototype._processLine.bind(this));
    } catch (e) {
        var err = e;
    } finally {
        cb(err);
    }
};

HexReader.prototype._transform = function (buff, enc, cb) {
    var lines = buff.toString('ascii').split('\n');
    lines[0] = this._line + lines[0];   // previous chunk had start of our first line
    this._line = lines.pop();           // our last line is always incomplete (or `=== ''`)
    this._processLines(lines, cb);
};

HexReader.prototype._flush = function (cb) {
    if (this._line) this._processLines([this._line], cb);
    else cb();
};

function readLine(str) {
    if (str[str.length-1] === '\r') str = str.slice(0,-1);
    if (str[0] !== ':') throw Error("Unexpected prefix, line does not appear to contain HEX data.");
    if (str.len < 11) throw Error("Line too short to contain HEX data.");
    if (Buffer.isBuffer(str)) str = str.toString('ascii');
    
    // TODO: should just convert whole `str.slice(1)` to buffer instead of individual _unhex'ing!
    var reclen = _unhex(str.slice(1,3)),
        offset = _unhex(str.slice(3,7)),
        rectype = _unhex(str.slice(7,9)),
        data = new Buffer(str.slice(9,-2), 'hex'),
        chksum = _unhex(str.slice(-2));
    
    if (chksum !== checksum([reclen, (offset & 0xFF), (offset >> 8), rectype], data)) throw Error("Checksum mismatch!");
    if (reclen !== data.length) throw Error("Data size mismatch, expected "+reclen+" but only received "+data.length+" bytes.");
    return {offset:offset, rectype:rectype, data:data};
}


function HexWriter(opts) {
    opts || (opts = {});
    stream.Transform.call(this, {objectMode:true});
    this._inHB = null;      // active linear address prefix
    this._addr = opts.startAddress || 0;
    this._wdsz = opts.bytesPerAddr || 1;    // NOTE: Microchip's bootloader always uses 1-byte addressing *in HEX*
    this._llen = opts.lineLength || 64;
}
util.inherits(HexWriter, stream.Transform);

HexWriter.prototype._pushBuffer = function (buff) {
    var boff = 0;
    while (boff < buff.length) {
        var chunk = buff.slice(boff, boff += this._llen),
            addrHB = this._addr >> 16,
            addrLB = this._addr & 0xFFFF;
        if (addrHB !== this._inHB) {
            this.push(makeLine(0, 'LINEAR_ADDR', Buffer([addrHB >> 8, addrHB & 0xFF])));
            this._inHB = addrHB;
        }
        this.push(makeLine(addrLB, 'DATA', chunk));
        this._addr += chunk.length*this._wdsz;
    }
};

HexWriter.prototype._transform = function (obj, enc, cb) {
    if ('startAddress' in obj) this._addr = obj.startAddress;
    // NOTE: can push address or a Buffer, *or* a buffer with address
    if (Buffer.isBuffer(obj)) this._pushBuffer(obj);
    cb();
};

HexWriter.prototype._flush = function (cb) {
    this.push(makeLine(0, 'EOF', Buffer(0)));
    cb();
};

function makeLine(offset, rectype, data) {
    if (data.length > 0xFF) throw Error("Data is too long!");
    if (offset > 0XFFFF) throw Error("Offset address too large.");
    if (typeof rectype === 'string') rectype = RECTYPE[rectype];
    
    var reclen = data.length,
        chksum = checksum([reclen, (offset & 0xFF), (offset >> 8), rectype], data);
    return [':', _hex(reclen, 0xFF), _hex(offset, 0xFFFF), _hex(rectype, 0xFF), data.toString('hex'), _hex(chksum, 0xFF), '\r\n'].join('');
}


exports.HexReader = HexReader;
exports.HexWriter = HexWriter;
// HACK: exposed for friendly usage
exports._rangeForAddr = _rangeForAddr;
exports._RANGES = RANGES;
